/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: CPGPException.h,v 1.7 2001/01/25 22:10:46 jeffc Exp $
____________________________________________________________________________*/

#ifndef Included_CPGPException_h	/* [ */
#define Included_CPGPException_h

#include <string>
#include <exception>
#include "pgpPFLErrors.h"
#include "pgpTypes.h"


//	These functions must be defined locally in your project
PGPError	ConvertOSErrorToPGPError(PGPInt32 inError);
void		ConvertPGPErrorToString(PGPError inError, string * outString);

#if PGP_DEBUG
	void		DumpPGPDebugMessage(const char * inMessage);
#endif


// The exception classes
class CPGPException : public exception {
public:
	explicit				CPGPException(PGPError inError, const char * inErrorString = 0);
#if PGP_MACINTOSH
							CPGPException(PGPError inError, const uchar * inErrorString);
#endif
	virtual					~CPGPException();
	
	virtual void			SetError(PGPError inError, const char * inErrorString = 0);
	
	virtual const char *	what() const throw() { return GetErrorString(); }
	virtual PGPInt32		GetError() const { return PGPInt32(GetPGPError()); }
	
	bool					IsCancelError() const { return mError == kPGPError_UserAbort; }
	PGPError				GetPGPError() const { return mError; }
	const char *			GetErrorString() const { return mErrorString.c_str(); }
	
#if PGP_DEBUG
	virtual void			DebugMessage(const char * inFile, int inLine) const;
#endif
	
protected:
	PGPError				mError;
	string					mErrorString;
	
	explicit				CPGPException() { }
};



class COSException : public CPGPException {
public:
	explicit				COSException(PGPInt32 inError, const char * inErrorString = 0);
#if PGP_MACINTOSH
							COSException(PGPInt32 inError, const uchar * inErrorString);
#endif
	virtual					~COSException();
	
	virtual PGPInt32		GetError() const { return GetOSError(); }
	
	PGPInt32				GetOSError() const { return mOSError; }

#if PGP_DEBUG
	virtual void			DebugMessage(const char * inFile, int inLine) const;
#endif

protected:
	PGPInt32				mOSError;
};



//	Create a stack based instance of this within your catch clause to convert all exceptions
//	into CPGPExceptions
class StPGPExceptionConverter {
public:
	explicit		StPGPExceptionConverter();
	virtual			~StPGPExceptionConverter();
	
	void			SetError(PGPError inError, const char * inErrorString = 0)
						{ mException->SetError(inError, inErrorString); }
	PGPInt32		GetError() const { return mException->GetError(); }
	
	bool			IsCancelError() const { return mException->IsCancelError(); }
	PGPError		GetPGPError() const { return mException->GetPGPError(); }
	const char *	GetErrorString() const { return mException->GetErrorString(); }
	
					operator CPGPException&() { return *mException; }
	
protected:
	CPGPException *	mException;
	
	virtual void	Convert();
};



// Utilities
#if PGP_DEBUG
#	define PGPThrowPGPError_(inError)									\
		do {															\
			CPGPException	__theException(inError);					\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#	define PGPThrowPGPErrorString_(inError, inString)					\
		do {															\
			CPGPException	__theException((inError), (inString));		\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#	define PGPThrowOSError_(inError)									\
		do {															\
			COSException	__theException(inError);					\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#	define PGPThrowOSErrorString_(inError, inString)					\
		do {															\
			COSException	__theException((inError), (inString));		\
			__theException.DebugMessage(__FILE__, __LINE__);			\
			throw __theException;										\
		} while (false)
#else
#	define PGPThrowPGPError_(inError)					throw CPGPException(inError)
#	define PGPThrowPGPErrorString_(inError, inString)	throw CPGPException((inError), (inString))
#	define PGPThrowOSError_(inError)					throw COSException(inError)
#	define PGPThrowOSErrorString_(inError, inString)	throw COSException((inError), (inString))
#endif



#define PGPThrowIfPGPError_(inError)									\
	do {																\
		PGPError	__theError = (inError);								\
		if (__theError != kPGPError_NoErr) {							\
			PGPThrowPGPError_(__theError);								\
		}																\
	} while (false)



#define PGPThrowIfOSError_(inError)										\
	do {																\
		OSStatus	__theError = (inError);								\
		if (__theError != 0) {											\
			PGPThrowOSError_(__theError);								\
		}																\
	} while (false)



#define PGPThrowIfErrorString_(inError, inString)						\
	do {																\
		OSStatus	__theError = (inError);								\
		if (__theError != kPGPError_NoErr) {							\
			PGPThrowErrorString_(__theError, inString);					\
		}																\
	} while (false)


#define PGPThrowIfOSErrorString_(inError, inString)						\
	do {																\
		OSStatus	__theError = (inError);								\
		if (__theError != 0) {											\
			PGPThrowOSErrorString_(__theError, inString);				\
		}																\
	} while (false)



#define PGPThrowIfNULL_(inPointer)										\
	if ((inPointer) == 0) {												\
		PGPThrowPGPError_(kPGPError_BadMemAddress);						\
	}



#define	PGPThrowIf_(inTest, inString)									\
		if (inTest) {													\
			PGPThrowPGPErrorString_(kPGPError_AssertFailed,				\
				(inString));											\
		}



#define	PGPThrowIfNot_(inTest, inString)								\
		if (! (inTest)) {												\
			PGPThrowPGPErrorString_(kPGPError_AssertFailed,				\
				(inString));											\
		}



#if PGP_MACINTOSH
#	include <Errors.h>

#	define PGPThrowIfResError_()	PGPThrowIfOSError_(ResError())
#	define PGPThrowIfMemError_()	PGPThrowIfOSError_(MemError())
#	define PGPThrowIfResFail_(inHandle)									\
		if ((inHandle) == 0) {											\
			OSStatus __theError = ResError();							\
			if (__theError == noErr) {									\
				__theError = resNotFound;								\
			}															\
			PGPThrowOSError_(__theError);								\
		}
#	define PGPThrowIfMemFail_(inPointer)								\
		if ((inPointer) == 0) {											\
			OSStatus __theError = MemError();							\
			if (__theError == noErr) {									\
				__theError = memFullErr;								\
			}															\
			PGPThrowOSError_(__theError);								\
		}
#endif


#endif /* Included_CPGPException_h */
